/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: OS_Win32.c $
   Last Modification:
    $Author: Robert $
    $Modtime: 28.04.09 14:49 $
    $Revision: 3128 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
     Win32 OS Abstraction Layer implementation. This MUST not be used in kernel mode
     drivers, but can be used in Userland, or Windows CE.
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     8         28.11.2011  RM       Added OS_Time() function
     7         14.06.2011  RM       Updated data type for OS_Init() for 32/64 Bit usage
     6         01.04.2010  RM       Data types changed for 32/64 Bit usage
     5         05.03.2010  RM       OS_CreateMutex set bInitialOwner to FALSE (Multithreading Issue)
     4         28.04.2009  RM       OS_FileOpen() file open flags changed from O_RDWR to
                                    O_RDONLY
     
     3         31.07.2008  RM       - OS_MapUserPointer() and OS_UnmapUserPointer() now
                                    have a new parameter pvOSDependent
                                    - Handling of OS_MapUserPointer() return value changed.
                                    NULL means mapping error.
                                    
                                    
     2         28.03.2007  RM       Mutex functions included
     
     1         07.08.2006  MT       initial version
     
**************************************************************************************/

/*****************************************************************************/
/*! \file OS_Win32.c
*    Win32 OS Abstraction Layer implementation. This MUST not be used in
*    kernel mode drivers, but can be used in Userland, or Windows CE.        */
/*****************************************************************************/

#include "OS_Dependent.h"
#include <stdlib.h>
#include <malloc.h>
#include <memory.h>
#include <Windows.h>
#include <sys/stat.h> 
#include <sys/types.h>
#include <io.h>
#include <FCNTL.H>
#include <stdio.h>

#ifdef CIFX_TOOLKIT_TIME
  #include <time.h>
#endif

/*****************************************************************************/
/*!  \addtogroup CIFX_TK_OS_ABSTRACTION Operating System Abstraction
*    \{                                                                      */
/*****************************************************************************/

/*****************************************************************************/
/*! Memory allocation function
*   \param ulSize    Length of memory to allocate
*   \return Pointer to allocated memory                                      */
/*****************************************************************************/
void* OS_Memalloc(uint32_t ulSize)
{
  return malloc(ulSize);
}

/*****************************************************************************/
/*! Memory freeing function
*   \param pvMem Memory block to free                                        */
/*****************************************************************************/
void OS_Memfree(void* pvMem)
{
  free(pvMem);
}

/*****************************************************************************/
/*! Memory reallocating function (used for resizing dynamic toolkit arrays)
*   \param pvMem     Memory block to resize
*   \param ulNewSize new size of the memory block
*   \return pointer to the resized memory block                              */
/*****************************************************************************/
void* OS_Memrealloc(void* pvMem, uint32_t ulNewSize)
{
  return realloc(pvMem, ulNewSize);
}

/*****************************************************************************/
/*! Memory setting
*   \param pvMem     Memory block
*   \param bFill     Byte to use for memory initialization
*   \param ulSize    Memory size for initialization)                         */
/*****************************************************************************/
void OS_Memset(void* pvMem, uint8_t bFill, uint32_t ulSize)
{
  memset(pvMem, bFill, ulSize);
}

/*****************************************************************************/
/*! Copy memory from one block to another
*   \param pvDest    Destination memory block
*   \param pvSrc     Source memory block
*   \param ulSize    Copy size in bytes                                      */
/*****************************************************************************/
void OS_Memcpy(void* pvDest, void* pvSrc, uint32_t ulSize)
{
  memcpy(pvDest, pvSrc, ulSize);
}

/*****************************************************************************/
/*! Compare two memory blocks
*   \param pvBuf1    First memory block
*   \param pvBuf2    Second memory block
*   \param ulSize    Compare size in bytes
*   \return 0 if both buffers are equal                                      */
/*****************************************************************************/
int OS_Memcmp(void* pvBuf1, void* pvBuf2, uint32_t ulSize)
{
  return memcmp(pvBuf1, pvBuf2, ulSize);
}

/*****************************************************************************/
/*! Move memory
*   \param pvDest    Destination memory
*   \param pvSrc     Source memory
*   \param ulSize    Size in byte to move                                    */
/*****************************************************************************/
void OS_Memmove(void* pvDest, void* pvSrc, uint32_t ulSize)
{
  memmove(pvDest, pvSrc, ulSize);
}


/*****************************************************************************/
/*! Sleep for a specific time
*   \param ulSleepTimeMs  Time in ms to sleep for                            */
/*****************************************************************************/
void OS_Sleep(uint32_t ulSleepTimeMs)
{
  Sleep(ulSleepTimeMs);
}

/*****************************************************************************/
/*! Retrieve a counter based on millisecond used for timeout monitoring
*   \return Current counter value (resolution of this value will influence
*           timeout monitoring in driver/toolkit functions(                  */
/*****************************************************************************/
uint32_t OS_GetMilliSecCounter(void)
{
  return GetTickCount();
}

/*****************************************************************************/
/*! Create an auto reset event
*   \return handle to the created event                                      */
/*****************************************************************************/
void* OS_CreateEvent(void)
{
  return (void*)CreateEvent(NULL, FALSE, FALSE, NULL);
}

/*****************************************************************************/
/*! Set an event
*   \param pvEvent Handle to event being signalled                           */
/*****************************************************************************/
void OS_SetEvent(void* pvEvent)
{
  SetEvent((HANDLE)pvEvent);
}

/*****************************************************************************/
/*! Reset an event
*   \param pvEvent Handle to event being reset                               */
/*****************************************************************************/
void OS_ResetEvent(void* pvEvent)
{
  ResetEvent((HANDLE)pvEvent);
}

/*****************************************************************************/
/*! Delete an event
*   \param pvEvent Handle to event being deleted                             */
/*****************************************************************************/
void OS_DeleteEvent(void* pvEvent)
{
  CloseHandle((HANDLE)pvEvent);
}

/*****************************************************************************/
/*! Wait for the signalling of an event
*   \param pvEvent   Handle to event being wait for
*   \param ulTimeout Timeout in ms to wait for event
*   \return 0 if event was signalled                                         */
/*****************************************************************************/
uint32_t OS_WaitEvent(void* pvEvent, uint32_t ulTimeout)
{
  return (WaitForSingleObject((HANDLE)pvEvent, ulTimeout) == WAIT_OBJECT_0) ? CIFX_EVENT_SIGNALLED : CIFX_EVENT_TIMEOUT;
}

/*****************************************************************************/
/*! Compare two ASCII string
*   \param pszBuf1   First buffer
*   \param pszBuf2   Second buffer
*   \return 0 if strings are equal                                           */
/*****************************************************************************/
int OS_Strcmp(const char* pszBuf1, const char* pszBuf2)
{
  return strcmp(pszBuf1, pszBuf2);
}

/*****************************************************************************/
/*! Compare characters of two strings without regard to case
*   \param pszBuf1   First buffer
*   \param pszBuf2   Second buffer
*   \param ulLen     Number of characters to compare
*   \return 0 if strings are equal                                           */
/*****************************************************************************/
int OS_Strnicmp(const char* pszBuf1, const char* pszBuf2, uint32_t ulLen)
{
  return _strnicmp(pszBuf1, pszBuf2, ulLen);
}  

/*****************************************************************************/
/*! Query the length of an ASCII string
*   \param szText    ASCII string
*   \return character count of szText                                        */
/*****************************************************************************/
int OS_Strlen(const char* szText)
{
  return (int)strlen(szText);
}

/*****************************************************************************/
/*! Copies one string to another monitoring the maximum length of the target
*   buffer.
*   \param szDest    Destination buffer
*   \param szSource  Source buffer
*   \param ulLength  Maximum length to copy
*   \return pointer to szDest                                                */
/*****************************************************************************/
char* OS_Strncpy(char* szDest, const char* szSource, uint32_t ulLength)
{
  return strncpy(szDest, szSource, ulLength);
}


/*****************************************************************************/
/*! Create an interrupt safe locking mechanism (Spinlock/critical section)
*   \return handle to the locking object                                     */
/*****************************************************************************/
void* OS_CreateLock(void)
{
  LPCRITICAL_SECTION ptCritSec = OS_Memalloc(sizeof(*ptCritSec));

  InitializeCriticalSection(ptCritSec);

  return ptCritSec;
}

/*****************************************************************************/
/*! Enter a critical section/spinlock
*   \param pvLock Handle to the locking object                               */
/*****************************************************************************/
void OS_EnterLock(void* pvLock)
{
  LPCRITICAL_SECTION ptCritSec = (LPCRITICAL_SECTION)pvLock;
  EnterCriticalSection(ptCritSec);
}

/*****************************************************************************/
/*! Leave a critical section/spinlock
*   \param pvLock Handle to the locking object                               */
/*****************************************************************************/
void OS_LeaveLock(void* pvLock)
{
  LPCRITICAL_SECTION ptCritSec = (LPCRITICAL_SECTION)pvLock;
  LeaveCriticalSection(ptCritSec);
}

/*****************************************************************************/
/*! Delete a critical section/spinlock object
*   \param pvLock Handle to the locking object being deleted                 */
/*****************************************************************************/
void OS_DeleteLock(void* pvLock)
{
  LPCRITICAL_SECTION ptCritSec = (LPCRITICAL_SECTION)pvLock;

  DeleteCriticalSection(ptCritSec);
  OS_Memfree(ptCritSec);
}

/*****************************************************************************/
/*! Create an Mutex object for locking code sections 
*   \return handle to the mutex object                                       */
/*****************************************************************************/
void* OS_CreateMutex(void)
{
  return CreateMutex( NULL,     /* LPSECURITY_ATTRIBUTES lpMutexAttributes*/
                      FALSE,    /* BOOL bInitialOwner,*/
                      NULL);    /* LPCTSTR lpName */
}

/*****************************************************************************/
/*! Wait for mutex
*   \param pvMutex    Handle to the Mutex locking object
*   \param ulTimeout  Wait timeout                                           
*   \return !=0 on succes                                                    */
/*****************************************************************************/
int OS_WaitMutex(void* pvMutex, uint32_t ulTimeout)
{
  return (WaitForSingleObject((HANDLE)pvMutex, ulTimeout) == WAIT_OBJECT_0) ? 1 : 0;
}

/*****************************************************************************/
/*! Release a mutex section section
*   \param pvMutex Handle to the locking object                              */
/*****************************************************************************/
void OS_ReleaseMutex(void* pvMutex)
{
  ReleaseMutex( (HANDLE) pvMutex);
}

/*****************************************************************************/
/*! Delete a Mutex object
*   \param pvMutex Handle to the mutex object being deleted                  */
/*****************************************************************************/
void OS_DeleteMutex(void* pvMutex)
{
  CloseHandle( (HANDLE) pvMutex);
}

/*****************************************************************************/
/*! Opens a file in binary mode
*   \param szFile     Full file name of the file to open
*   \param pulFileLen Returned length of the opened file
*   \return handle to the file, NULL mean file could not be opened           */
/*****************************************************************************/
void* OS_FileOpen(char* szFile, uint32_t* pulFileLen)
{
  int   iFile = _open(szFile, _O_RDONLY| _O_BINARY);
  void* pvRet = NULL;

  if(iFile != -1)
  {
    struct _stat tBuffer = {0};

    if(_fstat(iFile, &tBuffer) != -1)
    {
      *pulFileLen = tBuffer.st_size;
      pvRet = (void*)iFile;
    }
  }

  return pvRet;
}

/*****************************************************************************/
/*! Closes a previously opened file
*   \param pvFile Handle to the file being closed                            */
/*****************************************************************************/
void OS_FileClose(void* pvFile)
{
  int iFile = (int)pvFile;
  _close(iFile);
}

/*****************************************************************************/
/*! Read a specific amount of bytes from the file
*   \param pvFile   Handle to the file being read from
*   \param ulOffset Offset inside the file, where read starts at
*   \param ulSize   Size in bytes to be read
*   \param pvBuffer Buffer to place read bytes in
*   \return number of bytes actually read from file                          */
/*****************************************************************************/
uint32_t OS_FileRead(void* pvFile, uint32_t ulOffset, uint32_t ulSize, void* pvBuffer)
{
  int           iFile = (int)pvFile;
  uint32_t ulRet = 0;

  if(ulOffset == (uint32_t)_lseek(iFile, ulOffset, SEEK_SET))
  {
    ulRet = _read(iFile, pvBuffer, ulSize);
  }

  return ulRet;
}

/*****************************************************************************/
/*! OS specific initialization (if needed), called during cifXTKitInit()     */
/*****************************************************************************/
int32_t OS_Init()
{
  return 0;
}

/*****************************************************************************/
/*! OS specific de-initialization (if needed), called during cifXTKitInit()  */
/*****************************************************************************/
void OS_Deinit()
{
}

/*****************************************************************************/
/*! This functions is called for PCI cards in the toolkit. It is expected to
* write back all BAR's (Base address registers), Interrupt and Command
* Register. These registers are invalidated during cifX Reset and need to be
* re-written after the reset has succeeded
*   \param pvOSDependent OS Dependent Variable passed during call to
*                        cifXTKitAddDevice
*   \param pvPCIConfig   Configuration returned by OS_ReadPCIConfig
*                        (implementation dependent)                          */
/*****************************************************************************/
void OS_WritePCIConfig(void* pvOSDependent, void* pvPCIConfig)
{
  /* TODO: Implement PCI register accesss, needed for cifX cards, This cannot be done
           in user mode and needs to be moved to kernel mode */
  UNREFERENCED_PARAMETER(pvOSDependent);
  UNREFERENCED_PARAMETER(pvPCIConfig);
}

/*****************************************************************************/
/*! This functions is called for PCI cards in the toolkit. It is expected to
* read all BAR's (Base address registers), Interrupt and Command Register.
* These registers are invalidated during cifX Reset and need to be
* re-written after the reset has succeeded
*   \param pvOSDependent OS Dependent Variable passed during call to
*                        cifXTKitAddDevice
*   \return pointer to stored register copies (implementation dependent)     */
/*****************************************************************************/
void* OS_ReadPCIConfig(void* pvOSDependent)
{
  /* TODO: Implement PCI register accesss, needed for cifX cards, This cannot be done
           in user mode and needs to be moved to kernel mode */
  UNREFERENCED_PARAMETER(pvOSDependent);

  return NULL;
}

/*****************************************************************************/
/*! This function Maps a DPM pointer to a user application if needed.
*   This example just returns the pointer valid inside the driver.
*   In Win32 Kernel Mode this must be replaced by system calls
*   \param pvDriverMem   Pointer to be mapped
*   \param ulMemSize     Size to be mapped
*   \param ppvMappedMem  Returned mapped pointer (usable by application)
*   \param pvOSDependent OS Dependent variable passed during call to
*                        cifXTKitAddDevice
*   \return Handle that is needed for unmapping NULL is a mapping failure    */
/*****************************************************************************/
void* OS_MapUserPointer(void* pvDriverMem, uint32_t ulMemSize, void** ppvMappedMem, void* pvOSDependent)
{
  *ppvMappedMem = pvDriverMem;

  UNREFERENCED_PARAMETER(ulMemSize);
  UNREFERENCED_PARAMETER(pvOSDependent); 

  return pvDriverMem;
}

/*****************************************************************************/
/*! This function unmaps a previously mapped user application pointer 
*   In Win32 Kernel Mode this function must call
*   \param phMapping      Handle that was returned by OS_MapUserPointer
*   \param pvOSDependent  OS Dependent variable passed during call to
*                         cifXTKitAddDevice
*   \return !=0 on success                                                   */
/*****************************************************************************/
int OS_UnmapUserPointer(void* phMapping, void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(phMapping);
  UNREFERENCED_PARAMETER(pvOSDependent);

  return 1;
}

/*****************************************************************************/
/*! This function enables the interrupts for the device physically
*   \param pvOSDependent OS Dependent Variable passed during call to
*                        cifXTKitAddDevice                                   */
/*****************************************************************************/
void OS_EnableInterrupts(void* pvOSDependent)
{
  printf("We do not support IRQs in Win32 User mode!!");

  UNREFERENCED_PARAMETER(pvOSDependent);
}

/*****************************************************************************/
/*! This function enables the interrupts for the device physically
*   \param pvOSDependent OS Dependent Variable passed during call to
*                        cifXTKitAddDevice                                   */
/*****************************************************************************/
void OS_DisableInterrupts(void* pvOSDependent)
{
  printf("We do not support IRQs in Win32 User mode!!");

  UNREFERENCED_PARAMETER(pvOSDependent);
}


#ifdef CIFX_TOOLKIT_TIME
/*****************************************************************************/
/*! Get System time
*   \param ptTime   Pointer to store the time value
*   \return actual time value
/*****************************************************************************/
uint32_t OS_Time( uint32_t *ptTime)
{
  /* This function must deliver a one second counter since 1970/01/01, UTC time zone */
  time_t tTime = time(NULL);

  if( NULL != ptTime)
    *ptTime = (uint32_t) tTime;
  
  return (uint32_t)tTime;
}
#endif

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/
